<?php

namespace App\Filament\Resources\ContaPagars\Tables;

use Carbon\Carbon;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\BadgeColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Enums\FiltersLayout;
use Illuminate\Database\Eloquent\Builder;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use App\Models\Fornecedor;

class ContaPagarsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(function (Builder $query) {
                // Padrão: dados do mês corrente
                $query->whereMonth('data_emissao', now()->month)
                      ->whereYear('data_emissao', now()->year);
            })
            ->columns([
                TextColumn::make('fornecedor.nome')
                    ->label('Fornecedor')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('descricao')
                    ->searchable()
                    ->limit(50),
                TextColumn::make('valor')
                    ->numeric()
                    ->sortable()
                    ->suffix(' MT')
                    ->summarize([
                        \Filament\Tables\Columns\Summarizers\Sum::make()
                            ->suffix(' MT')
                            ->label('Total'),
                    ]),
                TextColumn::make('data_emissao')
                    ->date('d/m/Y')
                    ->sortable(),
                TextColumn::make('data_vencimento')
                    ->date('d/m/Y')
                    ->sortable(),
                TextColumn::make('data_pagamento')
                    ->date('d/m/Y')
                    ->sortable()
                    ->toggleable(),
                TextColumn::make('valor_pago')
                    ->numeric()
                    ->sortable()
                    ->suffix(' MT')
                    ->toggleable()
                    ->summarize([
                        \Filament\Tables\Columns\Summarizers\Sum::make()
                            ->suffix(' MT')
                            ->label('Total Pago'),
                    ]),
                BadgeColumn::make('status')
                    ->searchable()
                    ->formatStateUsing(function ($state, $record) {
                        if ($record->status !== 'pago' && $record->data_vencimento && Carbon::parse($record->data_vencimento)->isPast()) {
                            $vencido = (int) Carbon::parse($record->data_vencimento)->diffInDays(Carbon::now(), false);
                            if ($vencido > 0) {
                                return "Vencido há {$vencido} dias";
                            }
                            if ($vencido === 0) {
                                return "Vence hoje";
                            }
                        } elseif ($record->status !== 'pago' && $record->data_vencimento && Carbon::parse($record->data_vencimento)->isFuture()) {
                            $diasRestantes = (int) Carbon::now()->diffInDays(Carbon::parse($record->data_vencimento), false);
                            if ($diasRestantes <= 7) {
                                return "Vence em {$diasRestantes} dias";
                            }
                            return 'Dentro do prazo';
                        }
                        return ucfirst($state);
                    })
                    ->color(function ($state, $record) {
                        if ($record->status !== 'pago' && $record->data_vencimento && Carbon::parse($record->data_vencimento)->isPast()) {
                            return 'danger'; // Vermelho - vencido
                        } elseif ($record->status !== 'pago' && $record->data_vencimento && Carbon::parse($record->data_vencimento)->isFuture()) {
                            $diasRestantes = (int) Carbon::now()->diffInDays(Carbon::parse($record->data_vencimento), false);
                            if ($diasRestantes <= 7) {
                                return 'warning'; // Amarelo - próximo ao vencimento
                            }
                            return 'info'; // Azul - dentro do prazo
                        } elseif ($record->status == 'pago') {
                            return 'success'; // Verde - pago
                        }
                        return 'gray';
                    }),
                TextColumn::make('forma_pagamento')
                    ->searchable()
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'dinheiro' => 'success',
                        'cartao_credito' => 'warning',
                        'cartao_debito' => 'info',
                        'pix' => 'primary',
                        'boleto' => 'danger',
                        'cheque' => 'gray',
                        'transferencia' => 'info',
                        default => 'gray',
                    }),
                TextColumn::make('centro_custo.nome')
                    ->searchable()
                    ->toggleable(),
                TextColumn::make('created_at')
                    ->dateTime('d/m/Y H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('updated_at')
                    ->dateTime('d/m/Y H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('deleted_at')
                    ->dateTime('d/m/Y H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                // FILTRO POR FORNECEDOR
                Filter::make('fornecedor')
                    ->form([
                        Select::make('fornecedor_id')
                            ->label('Fornecedor')
                            ->placeholder('Selecione um fornecedor')
                            ->options(function () {
                                return Fornecedor::query()
                                    ->orderBy('nome')
                                    ->pluck('nome', 'id')
                                    ->toArray();
                            })
                            ->searchable()
                            ->preload(),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->when(
                            $data['fornecedor_id'] ?? null,
                            fn (Builder $query, $fornecedorId): Builder => $query->where('fornecedor_id', $fornecedorId),
                        );
                    })
                    ->indicateUsing(function (array $data): ?string {
                        if (!($data['fornecedor_id'] ?? null)) {
                            return null;
                        }
                        
                        $fornecedor = Fornecedor::find($data['fornecedor_id']);
                        return $fornecedor ? 'Fornecedor: ' . $fornecedor->nome : null;
                    }),
                
                // Filtro por período de emissão
                Filter::make('data_emissao')
                    ->form([
                        DatePicker::make('data_inicio')
                            ->label('Data Início')
                            ->default(now()->startOfMonth()),
                        DatePicker::make('data_fim')
                            ->label('Data Fim')
                            ->default(now()->endOfMonth()),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['data_inicio'],
                                fn (Builder $query, $date): Builder => $query->whereDate('data_emissao', '>=', $date),
                            )
                            ->when(
                                $data['data_fim'],
                                fn (Builder $query, $date): Builder => $query->whereDate('data_emissao', '<=', $date),
                            );
                    })
                    ->indicateUsing(function (array $data): array {
                        $indicators = [];
                        if ($data['data_inicio'] ?? null) {
                            $indicators[] = 'Data início: ' . Carbon::parse($data['data_inicio'])->format('d/m/Y');
                        }
                        if ($data['data_fim'] ?? null) {
                            $indicators[] = 'Data fim: ' . Carbon::parse($data['data_fim'])->format('d/m/Y');
                        }
                        return $indicators;
                    }),
                
                // Filtro por status
                SelectFilter::make('status')
                    ->options([
                        'pago' => 'Pago',
                        'pendente' => 'Pendente',
                        'cancelado' => 'Cancelado',
                    ])
                    ->multiple(),
                
                // Filtro por forma de pagamento
                SelectFilter::make('forma_pagamento')
                    ->options([
                        'dinheiro' => 'Dinheiro',
                        'cartao_credito' => 'Cartão de Crédito',
                        'cartao_debito' => 'Cartão de Débito',
                        'pix' => 'PIX',
                        'boleto' => 'Boleto',
                        'cheque' => 'Cheque',
                        'transferencia' => 'Transferência',
                    ])
                    ->multiple(),
                
                // Filtro por centro de custo
                SelectFilter::make('centro_custo_id')
                    ->label('Centro de Custo')
                    ->relationship('centro_custo', 'nome')
                    ->searchable()
                    ->preload(),
                
                // Filtro por vencimento
                Filter::make('vencimento')
                    ->form([
                        DatePicker::make('data_vencimento_inicio')
                            ->label('Vencimento Início'),
                        DatePicker::make('data_vencimento_fim')
                            ->label('Vencimento Fim'),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['data_vencimento_inicio'],
                                fn (Builder $query, $date): Builder => $query->whereDate('data_vencimento', '>=', $date),
                            )
                            ->when(
                                $data['data_vencimento_fim'],
                                fn (Builder $query, $date): Builder => $query->whereDate('data_vencimento', '<=', $date),
                            );
                    }),
                
                
                // Filtro rápido por situação do vencimento
                SelectFilter::make('situacao_vencimento')
                    ->label('Situação do Vencimento')
                    ->options([
                        'vencidos' => 'Vencidos',
                        'vencendo_hoje' => 'Vencem hoje',
                        'vencer_7_dias' => 'Vencem em até 7 dias',
                        'a_vencer' => 'A vencer',
                    ])
                    ->query(function (Builder $query, array $data) {
                        if (!isset($data['value'])) {
                            return $query;
                        }
                        
                        $hoje = Carbon::now();
                        
                        return match ($data['value']) {
                            'vencidos' => $query
                                ->where('status', '!=', 'pago')
                                ->whereDate('data_vencimento', '<', $hoje),
                            'vencendo_hoje' => $query
                                ->where('status', '!=', 'pago')
                                ->whereDate('data_vencimento', $hoje),
                            'vencer_7_dias' => $query
                                ->where('status', '!=', 'pago')
                                ->whereDate('data_vencimento', '>', $hoje)
                                ->whereDate('data_vencimento', '<=', $hoje->copy()->addDays(7)),
                            'a_vencer' => $query
                                ->where('status', '!=', 'pago')
                                ->whereDate('data_vencimento', '>', $hoje),
                            default => $query,
                        };
                    }),
                
                // Filtro por mês de vencimento
                SelectFilter::make('mes_vencimento')
                    ->label('Mês de Vencimento')
                    ->options([
                        'current' => 'Mês Corrente',
                        'next' => 'Próximo Mês',
                        'last' => 'Mês Passado',
                    ])
                    ->default('current')
                    ->query(function (Builder $query, array $data) {
                        if (!isset($data['value']) || $data['value'] === 'current') {
                            return $query
                                ->whereMonth('data_vencimento', now()->month)
                                ->whereYear('data_vencimento', now()->year);
                        }
                        
                        return match ($data['value']) {
                            'last' => $query
                                ->whereMonth('data_vencimento', now()->subMonth()->month)
                                ->whereYear('data_vencimento', now()->subMonth()->year),
                            'next' => $query
                                ->whereMonth('data_vencimento', now()->addMonth()->month)
                                ->whereYear('data_vencimento', now()->addMonth()->year),
                            default => $query,
                        };
                    }),
            ])
            ->filtersLayout(FiltersLayout::AboveContent)
            ->defaultSort('data_vencimento', 'asc')
            ->actions([ // CORRIGIDO: recordActions -> actions
                ViewAction::make(),
                EditAction::make(),
            ])
            ->bulkActions([ // CORRIGIDO: toolbarActions -> bulkActions
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ])
            ->searchable()
            ->persistSearchInSession()
            ->persistColumnSearchesInSession()
            ->persistFiltersInSession();
    }
}