<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Facades\Log;

class LancamentoContabil extends Model
{
    
    use HasFactory;
use SoftDeletes;
    protected $fillable=[
        'tipo',
'cliente_id',
'fornecedor_id',
'conta_id',
'frete_id',
'factura_id',
'conta_receber_id',
'descricao',
'valor',
'data_emissao',
'data_pagamento',
'forma_pagamento',
'centro_custo_id',
'observacoes'
            
    ];
       public function cliente()
      {
        return $this->belongsTo(Cliente::class);
      }

         public function frete()
      {
        return $this->belongsTo(Frete::class);
      }

      public function fornecedor()
      {
        return $this->belongsTo(Fornecedor::class);
      }

         public function custo_operacional()
      {
        return $this->belongsTo(CustoOperacional::class);
      }

        public function centro_custo()
      {
        return $this->belongsTo(CentroCusto::class);
      }

    public function factura()
      {
        return $this->belongsTo(Factura::class);
      }

      public function conta()
      {
        return $this->belongsTo(Conta::class);
      }

         public function conta_receber()
      {
        return $this->belongsTo(ContaReceber::class);
      }

      protected $casts = [
        'data_emissao' => 'date',
        'data_pagamento' => 'date',
    ];

    protected static function booted()
    {
        // Quando criar um novo lançamento
        static::created(function ($lancamento) {
            Log::info('Lançamento criado', ['id' => $lancamento->id, 'tipo' => $lancamento->tipo, 'fornecedor_id' => $lancamento->fornecedor_id]);
            
            if ($lancamento->tipo === 'entrada' && $lancamento->fornecedor_id) {
                Log::info('Vai criar IVA para lançamento', ['id' => $lancamento->id]);
                $lancamento->criarOuAtualizarContaPagarIva();
            } else {
                Log::info('Não vai criar IVA - condições não atendidas', [
                    'tipo' => $lancamento->tipo, 
                    'fornecedor_id' => $lancamento->fornecedor_id
                ]);
            }
        });

        // Quando atualizar um lançamento
        static::updated(function ($lancamento) {
            Log::info('Lançamento atualizado', ['id' => $lancamento->id]);
            
            // Verificar se os campos relevantes foram alterados
            $camposRelevantes = ['tipo', 'valor', 'fornecedor_id', 'descricao', 'data_emissao', 'forma_pagamento', 'centro_custo_id'];
            $camposAlterados = $lancamento->getChanges();
            
            Log::info('Campos alterados', $camposAlterados);
            
            // Buscar conta a pagar existente
            $contaPagar = ContaPagar::where('lancamento_origem_id', $lancamento->id)->first();
            Log::info('Conta pagar existente', ['existe' => $contaPagar ? 'sim' : 'nao', 'id' => $contaPagar?->id]);
            
            if ($lancamento->tipo === 'entrada' && $lancamento->fornecedor_id) {
                Log::info('É saída com fornecedor');
                
                if ($contaPagar) {
                    Log::info('Atualizando conta pagar existente');
                    $lancamento->atualizarContaPagarIva($contaPagar);
                } else {
                    Log::info('Criando nova conta pagar');
                    $lancamento->criarOuAtualizarContaPagarIva();
                }
            } else {
                Log::info('Não é saída ou não tem fornecedor');
                if ($contaPagar) {
                    Log::info('Removendo conta pagar existente');
                    $contaPagar->delete();
                }
            }
        });
    }

    /**
     * Criar ou atualizar conta a pagar de IVA
     */
    public function criarOuAtualizarContaPagarIva()
    {
        try {
            $valorIva = $this->valor * 0.16; // 16% de IVA
            
            Log::info('Valor IVA calculado', ['valor_original' => $this->valor, 'valor_iva' => $valorIva]);
            
            // Verificar se já existe uma conta a pagar para este lançamento
            $contaPagar = ContaPagar::where('lancamento_origem_id', $this->id)->first();
            
            if ($contaPagar) {
                Log::info('Atualizando conta pagar existente', ['id' => $contaPagar->id]);
                return $this->atualizarContaPagarIva($contaPagar);
            } else {
                Log::info('Criando nova conta pagar');
                
                $dataVencimento = $this->calcularDataVencimentoIva();
                Log::info('Data vencimento', ['data' => $dataVencimento]);
                
                $contaPagar = ContaPagar::create([
                    'lancamento_origem_id' => $this->id,
                    'fornecedor_id' => $this->fornecedor_id,
                    'descricao' => 'IVA 16% referente ao lançamento: ' . $this->descricao,
                    'valor' => $valorIva,
                    'data_emissao' => $this->data_emissao,
                    'data_vencimento' => $dataVencimento,
                    'status' => 'pendente',
                    'forma_pagamento' => $this->forma_pagamento,
                    'centro_custo_id' => $this->centro_custo_id,
                    'observacoes' => 'IVA gerado automaticamente do lançamento #' . $this->id,
                ]);
                
                Log::info('Conta pagar criada', ['id' => $contaPagar->id]);
                return $contaPagar;
            }
        } catch (\Exception $e) {
            Log::error('Erro ao criar/atualizar conta pagar', [
                'mensagem' => $e->getMessage(),
                'arquivo' => $e->getFile(),
                'linha' => $e->getLine()
            ]);
            return null;
        }
    }

    /**
     * Atualizar conta a pagar existente
     */
    public function atualizarContaPagarIva($contaPagar)
    {
        try {
            $valorIva = $this->valor * 0.16;
            
            $contaPagar->update([
                'fornecedor_id' => '1',
                'descricao' => 'IVA 16% referente ao lançamento: ' . $this->descricao,
                'valor' => $valorIva,
                'data_emissao' => $this->data_emissao,
                'data_vencimento' => $this->calcularDataVencimentoIva(),
                'status' => 'pendente',
                'forma_pagamento' => $this->forma_pagamento,
                'observacoes' => 'IVA atualizado do lançamento #' . $this->id,
            ]);
            
            Log::info('Conta pagar atualizada', ['id' => $contaPagar->id]);
            return $contaPagar;
        } catch (\Exception $e) {
            Log::error('Erro ao atualizar conta pagar', [
                'mensagem' => $e->getMessage(),
                'arquivo' => $e->getFile(),
                'linha' => $e->getLine()
            ]);
            return null;
        }
    }

    /**
     * Calcular data de vencimento do IVA
     */
  private function calcularDataVencimentoIva()
{
    $dataEmissao = Carbon::parse($this->data_emissao);
    
    // Retorna o último dia do mês da data de emissão
    return $dataEmissao->copy()->endOfMonth();
}

    /**
     * Relacionamento com conta a pagar de IVA
     */
    public function contaPagarIva()
    {
        return $this->hasOne(ContaPagar::class, 'lancamento_origem_id');
    }
}
